implementation module picture

//
//  Drawing functions and other operations on Pictures. 
//

import clCrossCall
      
::  * Picture :== (!HDC, !OS)

   
PackPicture :: !HDC !OS -> Picture
PackPicture hdc os = (hdc,os)

UnpackPicture :: !Picture -> (!HDC,!OS)
UnpackPicture p = p

::  PictureState = { ppensize   :: !Int,
                     ppenmode   :: !Int,
					 ppencolor  :: !(!Int,!Int,!Int),
					 pbackcolor :: !(!Int,!Int,!Int),
					 ppoint     :: !(!Int,!Int),
					 pfont      :: !Fnt
				   }

InitialPictureState :: PictureState
InitialPictureState  =  {  ppensize   =  1,
 						   ppenmode   =  iModeCopy,
						   ppencolor  =  (0,0,0),
						   pbackcolor =  (255,255,255),
						   ppoint     =  (0,0),
						   pfont      =  DefFnt
					  	}


::  UpdateArea      :== [Rectangle]
::  DrawFunction    :== Picture -> Picture

DoDrawFunctions :: ![DrawFunction] !Picture -> Picture				
DoDrawFunctions []       pic  =  pic
DoDrawFunctions [f:rest] pic  =  DoDrawFunctions rest (f pic)

 
ClipPicture ::  !Rectangle !Picture -> Picture
ClipPicture r pic =  WinClipPicture (CorrectRect r) pic


//  The predefined figures that can be drawn:

::  Point               :== (!Int, !Int)
::  Line                :== (!Point, !Point)
::  Curve               :== (!Oval, !Int, !Int)
::  Rectangle			:== (!Point, !Point)
::  RoundRectangle		:== (!Rectangle, !Int, !Int)
::  Oval                :== Rectangle
::  Circle			    :== (!Point, !Int)
::  Wedge               :== (!Oval, !Int, !Int)
::  Polygon				:== (!Point, !PolygonShape)

::  PolygonShape		:== [Vector]
::  Vector				:== (!Int, !Int)

//  The pen attributes which influence the way figures are drawn:

::  PenSize     :== (!Int, !Int)

::  PenMode    =  CopyMode    | OrMode    | XorMode    | ClearMode   | HiliteMode
                | NotCopyMode | NotOrMode | NotXorMode | NotClearMode

::  PenPattern  =  BlackPattern
                 | DkGreyPattern
                 | GreyPattern
                 | LtGreyPattern
                 | WhitePattern

//  The colours:

::  Colour  =  RGB Real Real Real
              |   BlackColour | RedColour
              |   WhiteColour | GreenColour
              |   BlueColour  | YellowColour
              |   CyanColour  | MagentaColour

     
MinRGB  :== 0.0
MaxRGB  :== 1.0

    
/*  Rules setting the attributes of a Picture:
*/

/*  SetPenSize (w, h) sets the PenSize to w pixels wide and h pixels high.
    SetPenMode sets the interference how new figures 'react' to drawn ones.
    SetPenPattern sets the way new figures are drawn.
    SetPenNormal sets the SetPenSize to (1,1), the PenMode to CopyMode and
        the PenPattern to BlackPattern. */

SetPenSize  ::  !PenSize        !Picture -> Picture
SetPenSize (w,h) pic =  WinSetPenSize ((w+h)/2) pic 


SetPenMode  ::  !PenMode        !Picture -> Picture
SetPenMode mode pic = WinSetMode imode pic
where
  imode = case mode of 
			 CopyMode		-> iModeCopy
			 OrMode			-> iModeOr
			 XorMode		-> iModeXor
			 ClearMode		-> iModeOr
			 HiliteMode		-> iModeXor
             NotCopyMode	-> iModeOr
			 NotOrMode		-> iModeOr
			 NotXorMode		-> iModeOr
			 NotClearMode	-> iModeOr
			


SetPenPattern ::  !PenPattern !Picture -> Picture
SetPenPattern pat pict = pict

SetPenNormal  :: !Picture -> Picture
SetPenNormal pic = newpic
where
  pic2    = WinSetPenSize 1 pic
  pic3    = WinSetMode iModeCopy pic2
  newpic  = SetPenPattern BlackPattern pic3

/*  Using colours:
    There are basically two types of Colours: RGB and basic colours.
        An RGB colour defines the amount of red (r), green (g) and blue (b)
        in a certain colour by the tuple (r,g,b). These are REAL values and
        each of them must be between MinRGB and MaxRGB (0.0 and 1.0).
        The colour black is defined by (MinRGB, MinRGB, MinRGB) and white
        by (MaxRGB, MaxRGB, MaxRGB).
        Given a RGB colour, all amounts are adjusted between MinRGB and
        MaxRGB.
    Only FullColour windows can apply RGB colours. Applications that use
    these windows may not run on all computers (e.g. Macintosh Plus).
    A small set of basic colours is defined that can be used on all systems.
    
    SetPenColour    sets the colour of the pen.
    SetBackColour   sets the background colour. */

SetPenColour    ::  !Colour !Picture -> Picture
SetPenColour colour pic = WinSetPenColor (CorrectColor colour) pic


SetBackColour    ::  !Colour !Picture -> Picture
SetBackColour colour pic = WinSetBackColor (CorrectColor colour) pic


/*  Using fonts:
    The initial font of a Picture is 8 point MS SansSerif in PlainStyle.
    SetFont      sets a new complete Font in the Picture.
    SetFontName  sets a new font without changing the style or size.
    SetFontStyle sets a new style without changing font or size.
    SetFontSize  sets a new size without changing font or style.
                     The size is always adjusted between MinFontSize and
                     MaxFontSize (see deltaFont.dcl). */

SetFont :: !Font !Picture -> Picture
SetFont font pic = WinSetFont font pic

SetFontName :: !FontName !Picture -> Picture
SetFontName name pic = WinSetFontName name pic

SetFontStyle :: ![FontStyle] !Picture -> Picture
SetFontStyle styles pic = WinSetFontStyle (SStyle2IStyle styles) pic

SetFontSize :: !FontSize !Picture -> Picture
SetFontSize size pic = WinSetFontSize size pic



PictureCharWidth    :: !Char   !Picture -> (!Int,      !Picture)
PictureCharWidth c pic = WinGetPicCharWidth c pic

PictureStringWidth  :: !String !Picture -> (!Int,      !Picture)
PictureStringWidth  str pic = WinGetPicStringWidth str pic

PictureFontMetrics  :: !Picture -> (!FontInfo, !Picture)
PictureFontMetrics pic =  ((a,d,w,l), pic`)
where
  (a,d,w,l,pic`) = WinGetPicFontInfo pic


/*  Rules changing the position of the pen:
*/

/*  Absolute and relative pen move operations (without drawing).
*/

MovePenTo   ::  !Point  !Picture -> Picture
MovePenTo pt pic = WinMovePenTo pt pic
 
MovePen ::      !Vector !Picture -> Picture
MovePen  v pic = WinMovePen v pic

/*  Absolute and relative pen move operations (with drawing).
*/

LinePenTo   ::  !Point  !Picture -> Picture
LinePenTo pt pic = WinLinePenTo pt pic

LinePen ::      !Vector !Picture -> Picture
LinePen  v pic = WinLinePen v pic
    
/*  DrawChar (DrawString) draws the character (string) in the current font.
    The baseline of the characters is the y coordinate of the pen.
    The new position of the pen is directly after the character (string)
    including spacing.
*/

DrawChar    ::  !Char   !Picture -> Picture
DrawChar c pic = WinDrawChar (toInt c) pic

DrawString  ::  !String !Picture -> Picture
DrawString s p = WinDrawString s p

/*  Rules not changing the position of the pen after drawing:
*/

/*  Non plane figures:
        Draw(C)Point    draws the pixel (in the given colour) in the Picture.
        Draw(C)Line     draws the line  (in the given colour) in the Picture.
        Draw(C)     draws the curve (in the given colour) in the Picture.
                        A Curve is part of an Oval o starting from angle a
                        upto angle b (both of type INT in degrees modulo 360):
                            (o, a, b).
                        See Wedges for further information on the angles. */

DrawPoint   ::  !Point  !Picture -> Picture
DrawPoint pt pic = WinDrawPoint pt pic

DrawLine    ::      !Line       !Picture -> Picture
DrawLine (start, end) pic = WinDrawLine start end pic

DrawCurve   ::  !Curve  !Picture -> Picture
DrawCurve curve pic = WinDrawCurve rect r1 r2 pic
where
	(rect,r1,r2) = CalcRadials curve 

DrawCPoint  ::  !Point !Colour !Picture -> Picture
DrawCPoint pt col pic = WinDrawCPoint pt (CorrectColor col) pic

DrawCLine   ::  !Line    !Colour !Picture -> Picture
DrawCLine  (start, end) col pic = WinDrawCLine start end (CorrectColor col) pic

DrawCCurve  ::  !Curve !Colour !Picture -> Picture
DrawCCurve  curve col pic = WinDrawCCurve rect r1 r2 (CorrectColor col) pic
where
	(rect,r1,r2) = CalcRadials curve 


/*  A Rectangle is defined by two of its diagonal corner Points (a, b)
    with: a: (a_x, a_y),
            b: (b_x, b_y)
    such that a_x <> b_x and a_y <> b_y.
    In case either a_x = b_x or a_y = b_y, the Rectangle is empty.

    DrawRectangle       draws the edges of the rectangle.
    FillRectangle       draws the edges and interior of the rectangle.
    EraseRectangle      erases the edges and interior of the rectangle.
    InvertRectangle inverts the edges and interior of the rectangle.

    MoveRectangleTo moves the contents of the rectangle to a new top corner.
    MoveRectangle       moves the contents of the rectangle over the given vector.
    CopyRectangleTo copies the contents of the rectangle to a new top corner.
    CopyRectangle       copies the contents of the rectangle over the given vector. */

DrawRectangle   ::  !Rectangle !Picture -> Picture
DrawRectangle r pic = WinDrawRectangle (CorrectRect r) pic


FillRectangle   ::  !Rectangle !Picture -> Picture
FillRectangle r pic = WinFillRectangle (CorrectRect r) pic

EraseRectangle  ::  !Rectangle !Picture -> Picture
EraseRectangle r pic = WinEraseRectangle  (CorrectRect r) pic

InvertRectangle ::  !Rectangle !Picture -> Picture
InvertRectangle r pic = WinInvertRectangle (CorrectRect r) pic

MoveRectangleTo ::  !Rectangle !Point  !Picture -> Picture
MoveRectangleTo r p pic = WinMoveRectangleTo (CorrectRect r) p pic 

MoveRectangle   ::  !Rectangle !Vector !Picture -> Picture
MoveRectangle r v pic = WinMoveRectangle (CorrectRect r) v pic 

CopyRectangleTo ::  !Rectangle !Point  !Picture -> Picture
CopyRectangleTo r p pic = WinCopyRectangleTo (CorrectRect r) p pic 

CopyRectangle   ::  !Rectangle !Vector !Picture -> Picture
CopyRectangle r v pic = WinCopyRectangle (CorrectRect r) v pic 


/*  Rounded corner rectangles: a RoundRectangle with enclosing Rectangle
    r and corner curvatures x and y is defined by the tuple (r, x, y).
    x (y) defines the horizontal (vertical) diameter of the corner curves.
    x (y) is always adjusted between 0 and the width (height) of r.
    Note:   RoundRectangle (r, 0, 0) is the Rectangle r, 
            RoundRectangle (r, w, h) is the Oval r if w and h are the width
            and height of r.
*/

DrawRoundRectangle  ::  !RoundRectangle !Picture -> Picture
DrawRoundRectangle  rr pic = WinDrawRoundRectangle r w h pic
where (r,w,h) = CorrectRoundRect rr

FillRoundRectangle  ::  !RoundRectangle !Picture -> Picture
FillRoundRectangle  rr pic = WinFillRoundRectangle r w h pic
where (r,w,h) = CorrectRoundRect rr

EraseRoundRectangle ::  !RoundRectangle !Picture -> Picture
EraseRoundRectangle rr pic = WinEraseRoundRectangle r w h pic
where (r,w,h) = CorrectRoundRect rr

InvertRoundRectangle ::  !RoundRectangle !Picture -> Picture
InvertRoundRectangle rr pic = WinInvertRoundRectangle r w h pic
where (r,w,h) = CorrectRoundRect rr

/*  Ovals: an Oval is defined by its enclosing Rectangle.
    Note : the Oval of a square Rectangle is a Circle.
*/

DrawOval    ::  !Oval !Picture -> Picture
DrawOval r pic = WinDrawOval (CorrectRect r) pic

FillOval    ::  !Oval !Picture -> Picture
FillOval r pic = WinFillOval (CorrectRect r) pic

EraseOval   ::  !Oval !Picture -> Picture
EraseOval r pic = WinEraseOval (CorrectRect r) pic

InvertOval  ::  !Oval !Picture -> Picture
InvertOval r pic = WinInvertOval (CorrectRect r) pic


/*  Circles:    a Circle with center c (Point) and radius r (INT) is
                defined by the tuple (c, r). */

DrawCircle   ::  !Circle !Picture -> Picture
DrawCircle  (pt, r) pic = WinDrawCircle pt r pic

FillCircle   ::  !Circle !Picture -> Picture
FillCircle  (pt, r) pic = WinFillCircle pt r pic

EraseCircle  ::  !Circle !Picture -> Picture
EraseCircle  (pt, r) pic = WinEraseCircle pt r pic

InvertCircle ::  !Circle !Picture -> Picture
InvertCircle  (pt, r) pic = WinInvertCircle pt r pic


/*  Wedges: a Wedge is a pie part of an Oval o starting from angle a
    upto angle b (both of type INT in degrees modulo 360):
        (o, a, b).
    Angles are always taken counterclockwise, starting from 3 o'clock.
    So angle 0 is at 3 o'clock, angle 90 (-270) at 12 o'clock,
    angle 180 (-180) at 9 o'clock and angle 270 (-90) at 6 o'clock. */


DrawWedge   ::  !Wedge !Picture -> Picture
DrawWedge  wedge pic = WinDrawWedge rect r1 r2 pic
where
	(rect,r1,r2) = CalcRadials wedge 


FillWedge   ::  !Wedge !Picture -> Picture
FillWedge  wedge pic = WinFillWedge rect r1 r2 pic
where
	(rect,r1,r2) = CalcRadials wedge 

EraseWedge  ::  !Wedge !Picture -> Picture
EraseWedge  wedge pic = WinEraseWedge rect r1 r2 pic
where
	(rect,r1,r2) = CalcRadials wedge 

InvertWedge ::  !Wedge !Picture -> Picture
InvertWedge  wedge pic = WinInvertWedge rect r1 r2 pic
where
	(rect,r1,r2) = CalcRadials wedge 

/*  Polygons: a Polygon is a figure drawn by a number of lines without
    taking the pen of the Picture, starting from some Point p.
    The PolygonShape s (a list [v1,...,vN] of Vectors) defines how the
    Polygon is drawn:
        MoveTo p, DrawLine from v1 upto vN, DrawLineTo p to close it.
    So a Polygon with s [] is actually the Point p.
    
    ScalePolygon    by scale k sets shape [v1,...,vN] into [k*v1,...,k*vN].
                        Negative, as well as 0 are valid scales.
    MovePolygonTo   changes the starting point into the given Point and
    MovePolygon     moves the starting point by the given Vector.
*/

DrawPolygon ::      !Polygon !Picture   -> Picture
DrawPolygon poly (hdc, os) = WinDrawPolygon (hdc, newos)
where
  newos = TransferPolygon poly os


FillPolygon ::      !Polygon !Picture   -> Picture
FillPolygon poly (hdc, os) = WinFillPolygon (hdc, newos)
where
  newos = TransferPolygon poly os

ErasePolygon    ::  !Polygon !Picture   -> Picture
ErasePolygon poly (hdc, os) = WinErasePolygon (hdc, newos)
where
  newos = TransferPolygon poly os

InvertPolygon   ::  !Polygon !Picture   -> Picture
InvertPolygon poly (hdc, os) = WinInvertPolygon (hdc, newos)
where
  newos = TransferPolygon poly os


ScalePolygon    ::  !Int        !Polygon -> Polygon
ScalePolygon f (p,points) = (p, map (scalepoint f) points)
where
  scalepoint f (x, y) = (x*f, y*f)  

MovePolygonTo   ::  !Point  !Polygon -> Polygon
MovePolygonTo p1 (p,points) = (p1, points)

MovePolygon ::  !Vector !Polygon -> Polygon
MovePolygon (dx, dy) ((px, py), points) = ((px+dx, py+dy), points)



//--------------------------------------------------------------

CorrectRect :: !Rectangle -> Rect
CorrectRect ((l,t),(r,b)) = (l`,t`,r`,b`)
where
	(t`,b`) = if (t < b) (t,b) (b,t)
	(l`,r`) = if (l < r) (l,r) (r,l)


CorrectRoundRect :: !RoundRectangle -> (!Rect,!Int,!Int)
CorrectRoundRect (rect,w,h) = ((l,t,r,b),w`,h`)
where
	(l,t,r,b) = CorrectRect rect
	w`        = CorrectRound w (r-l)
    h`        = CorrectRound h (b-t)

	CorrectRound r e
	  | r < 0    = 0
	  | r > e/2  = e/2
	             = r

deg2rad x :==  0.01745329251994 * x


CalcRadials :: !(!Rectangle, !Int, !Int) -> (!Rect, !Point,!Point)
CalcRadials (rect,s,e) = ((l,t,r,b), radial1, radial2)
where
  (l,t,r,b) = CorrectRect rect
  cx      =  (r+l)/2
  cy      =  (b+t)/2
  rads    =  deg2rad (toReal s)
  rade    =  deg2rad (toReal e)
  radial1 =  (  cx + toInt( 20.0 * cos (rads) ),
                cy - toInt( 20.0 * sin (rads) )  )
  radial2 =  (  cx + toInt( 20.0 * cos (rade) ),
                cy - toInt( 20.0 * sin (rade) )  )

   


CorrectColor :: !Colour -> RGBcolor
CorrectColor BlackColour	= (   0,   0,   0 )
CorrectColor BlueColour		= (   0,   0, 255 )
CorrectColor GreenColour	= (   0, 255,   0 )
CorrectColor CyanColour     = (   0, 255, 255 )
CorrectColor RedColour		= ( 255,   0,   0 )
CorrectColor MagentaColour  = ( 255,   0, 255 )
CorrectColor YellowColour   = ( 255, 255,   0 )
CorrectColor WhiteColour    = ( 255, 255, 255 )
CorrectColor (RGB r g b)    = ( red, green, blue )
where
	red   =  To255 r
	green =  To255 g
	blue  =  To255 b

	To255 c
	 | c <= MinRGB = 0;
     | c >= MaxRGB = 255;
				   = toInt (c * 255.0)



TransferPolygon :: !Polygon !OS -> !OS
TransferPolygon (start, deltas) os = finalos
where
   os2      = WinStartPolygon (1 + length deltas) os
   os3     = WinAddPolygonPoint start os2
   finalos = CopyDeltas start deltas os3

   CopyDeltas :: Point [Vector] OS -> OS
   CopyDeltas p [] os       = os
   CopyDeltas (px,py) [(dx,dy):rest] os = CopyDeltas newpos rest newos
   where
     newpos = (px+dx, py+dy)
     newos  = WinAddPolygonPoint newpos os
    


/////////////////////////////////////
//
//	FontStuff
//


:: Font :== !( !FontName, !IFontStyle, !FontSize )

:: FontName   :== String
:: FontStyle  :== String
:: IFontStyle :== Int
:: FontSize   :== Int
:: FontInfo   :== (!Int,!Int,!Int,!Int)


SBold      :== "Bold"
SItalic    :== "Italic"
SUnderline :== "Underline"
SStrikeOut :== "Strike Out"

MinFontSize :== 4
MaxFontSize :== 128

/*	SelectFont creates the font as specified by the name, the stylistic
	variations and size. In case there are no FontStyles ([]), the font
	is selected without stylistic variations (i.e. in plain style).
	The size is always adjusted between MinFontSize and MaxFontSize.
	The boolean result is True in case this font is available and needn't
	be scaled. In case the font is not available, the default font is
	chosen in the indicated style and size. */
SelectFont :: !FontName ![FontStyle] !FontSize -> (!Bool, !Font)
SelectFont fname styles size = ( True, (fname, SStyle2IStyle styles, Points2Height size ))

/*	DefaultFont returns name, style and size of the default font. */
DefaultFont :: (!FontName, ![FontStyle], !FontSize) 
DefaultFont =  ("MS Sans Serif", [], 8)

DefFnt :: Fnt
DefFnt = ( "MS Sans Serif", 0, Points2Height 8 )

/*	FontNames	returns the FontNames of all available fonts. */
FontNames  :: [ FontName ]
FontNames = SortAndRemoveDuplicates unsortednames
where
  GetFontNamesCci = (CcRqGETFONTNAMES, 0,0,0,0,0,0)
  (_, unsortednames, _) = IssueCleanRequest FontnamesCallback GetFontNamesCci [] 8

  FontnamesCallback :: !CrossCallInfo ![FontName] OS -> (Bool, CrossCallInfo, [FontName], OS)
  FontnamesCallback (mess, textptr, _,_,_,_,_) names os = (True, Return0Cci, [ newname : names ], os`)
  where
    (newname, os`) = WinGetCString textptr os

/*	FontSizes	returns all FontSizes of a font that are available without scaling.
	In case the font is unavailable, the styles or sizes of the default font
	are returned. */
FontSizes :: !FontName -> [ FontSize ]
FontSizes fname = SortAndRemoveDuplicates unsortedsizes
where
  ( textptr, os` )        =  WinMakeCString fname 8 
  GetFontSizesCci         =  (CcRqGETFONTSIZES, textptr, 0,0,0,0,0 )
  (_,unsortedsizes, _) =  IssueCleanRequest FontSizesCallback GetFontSizesCci [] os`


  FontSizesCallback :: !CrossCallInfo ![FontSize] !OS -> (!Bool, !CrossCallInfo, ![FontSize], !OS)
  FontSizesCallback (mess, s, 0,_,_,_,_) sizes os = (True, Return0Cci, newsizes, os )
  where
     newsizes = if (pts >= MinFontSize && pts <= MaxFontSize)
	                [ pts : sizes ]
					sizes
	 pts = Height2Points s 
  FontSizesCallback (mess, s, _,_,_,_,_) sizes os = (True, Return0Cci, [MinFontSize..MaxFontSize], os )


Height2Points :: !Int -> Int
Height2Points h = toInt points
where
  dpi      =  toReal WinGetVertResolution
  phfactor =  dpi / 72.0
  points   =  toReal h / phfactor

Points2Height :: !Int -> Int
Points2Height p = toInt height
where
  dpi      =  toReal WinGetVertResolution
  phfactor =  dpi / 72.0
  height   =  toReal p * phfactor


SortAndRemoveDuplicates :: !u:[a] -> u:[a] | Ord a
SortAndRemoveDuplicates [e:es]	= insert e (SortAndRemoveDuplicates es)
where
  insert:: a !u:[a] -> u:[a] | Ord a
  insert a list=:[b:x]
		| a<b		= [a:list]
		| a > b     = [b:insert a x]
		// otherwise
					= list
  insert a []	= [a]
SortAndRemoveDuplicates [] = []
   
/*	FontStyles	returns the FontStyles of all available styles. */
FontStyles :: !FontName -> [ FontStyle ] 
FontStyles _ = [ SBold, SItalic, SUnderline, SStrikeOut ]


SStyle2IStyle :: ![ FontStyle ] -> IFontStyle
SStyle2IStyle styles = s2i styles 0
where
  s2i []                    i = i
  s2i [ SBold      : rest ] i = s2i rest (i bitor iBold)
  s2i [ SItalic    : rest ] i = s2i rest (i bitor iItalic)
  s2i [ SUnderline : rest ] i = s2i rest (i bitor iUnderline)
  s2i [ SStrikeOut : rest ] i = s2i rest (i bitor iStrikeOut)
  s2i [ otherstyle : rest ] i = s2i rest i

/*	FontMetrics yields the FontInfo in terms of pixels of a given Font. The FontInfo
	is a four-tuple (ascent, descent, max width, leading) which defines the metrics
	of a font:
		- ascent	is the height of the top most character measured from the base
		- descent	is the height of the bottom most character measured from the base
		- max width	is the width of the widest character including spacing
		- leading	is the vertical distance between two lines of the same font.
	The full height of a line is the sum of the ascent, descent and leading. */

FontMetrics :: !Font -> FontInfo
FontMetrics fnt = (a,d,w,l)
where
  (a,d,w,l,_) = WinGetFontInfo fnt 90


/*	FontCharWidth(s) (FontStringWidth(s)) return the width(s) in terms of pixels
	of given character(s) (string(s)) for a particular Font. */
FontCharWidth :: !Char !Font -> Int
FontCharWidth c f = w
where
  (w,_) = WinGetCharWidth c f 90

FontCharWidths :: ![Char] !Font -> [Int]
FontCharWidths cs f = [ FontCharWidth c f \\ c <- cs ]

 
FontStringWidth :: !String !Font -> Int
FontStringWidth s f = w
where
  (w,_) = WinGetStringWidth s f 990


FontStringWidths :: ![String] !Font -> [Int]
FontStringWidths ss f = [ FontStringWidth s f \\ s <- ss ]
 
